(*
This unit demonstrates a very basic scheme of saving calendar notes to an
.ini file.
*)
unit DemoUnit;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, Grids, WallCalendar, IniFiles, StdCtrls;

type
  TDemoForm = class(TForm)
    WallCalendar1: TWallCalendar;
    SaveButton: TButton;
    CloseButton: TButton;
    ClearButton: TButton;
    Label1: TLabel;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure SaveButtonClick(Sender: TObject);
    procedure CloseButtonClick(Sender: TObject);
    procedure WallCalendar1DoubleClickCalendar(Sender: TObject; ACol,
      ARow: Integer; CellDate: TDateTime);
    procedure FormShow(Sender: TObject);
    procedure ClearButtonClick(Sender: TObject);
    procedure WallCalendar1ChangeMonth(Sender: TObject;
      PreviousMonth: TCalendarMonth);
    procedure WallCalendar1ChangeYear(Sender: TObject;
      PreviousYear: Integer);
    procedure WallCalendar1DecreaseCalendar(Sender: TObject;
      PreviousMonth: TCalendarMonth; PreviousYear: Integer);
    procedure WallCalendar1IncreaseCalendar(Sender: TObject;
      PreviousMonth: TCalendarMonth; PreviousYear: Integer);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
  private
    { Private declarations }
  protected
    procedure LoadCalendarNotesFromIniFile;
    procedure SaveCalendarNotesToIniFile;
    procedure AddNoteToList(TheNote: string; TheCellDate: TDateTime);
    procedure ShowNotesForCurrentMonthAndYear;
  public
    { Public declarations }
    // The Application's initialization file
    AppIniFile: TIniFile;

    (*
    CalendarNotesList is a string list where the actual note is stored in
    the string portion, and each object is an integer reference to the integer
    portion of the associated TDateTime value.  Storing this in the objects
    property allows us to look up existing note values and update them.
    *)
    CalendarNotesList: TStringList;
  end;

var
  DemoForm: TDemoForm;

implementation

{$R *.dfm}

const
    DATE_NOTE_SECTION = 'DateNotes';

procedure TDemoForm.FormCreate(Sender: TObject);
    begin
    // Initialize SaveToFile.ini
    AppIniFile := TIniFile.Create(ChangeFileExt(ParamStr(0), '.ini'));

    CalendarNotesList := TStringList.Create;
    end;

procedure TDemoForm.FormShow(Sender: TObject);

    begin
    WallCalendar1.SetCalendarDate(Now);

    LoadCalendarNotesFromIniFile;
    ShowNotesForCurrentMonthAndYear;

    WallCalendar1.SetFocus;
    end;

procedure TDemoForm.FormClose(Sender: TObject; var Action: TCloseAction);

    begin
    // Save the list to the .ini file before exiting
    SaveCalendarNotesToIniFile;
    end;

procedure TDemoForm.FormDestroy(Sender: TObject);

    begin
    FreeAndNil(AppIniFile);
    FreeAndNil(CalendarNotesList);
    end;

procedure TDemoForm.LoadCalendarNotesFromIniFile;
    {
    This procedure reads SaveToFile.ini and loads all calendar notes into the
    CalendarNotesList string list.  All calendar notes are stored in the .ini
    file under the key 'DateNotes', with the integer portion of the TDateTime
    as the key, and the actual note as the value.  A sample SaveToFile.ini file:

    [DateNotes]
    38082=Today is April 5th, 2004
    38172=It's the 4th of July!
    }

    var
        i: integer;
        TheDate: TDateTime;
        DateNote: string;
        DateKeys: TStringList;

    begin
    CalendarNotesList.Clear;
    DateKeys := TStringList.Create;
    try
        with AppIniFile do begin
            ReadSection(DATE_NOTE_SECTION, DateKeys);
            for i := 0 to Pred(DateKeys.Count) do begin
                DateNote := Trim(ReadString(DATE_NOTE_SECTION, DateKeys[i], ''));
                if DateNote <> '' then begin
                    TheDate := StrToIntDef(DateKeys[i], 0);
                    if TheDate <> 0 then begin
                        CalendarNotesList.AddObject(DateNote, TObject(trunc(TheDate)));
                        end;
                    end;
                end;
            end;
    finally
        FreeAndNil(DateKeys);
        end;
    end;

procedure TDemoForm.SaveCalendarNotesToIniFile;
    {
    This procedure writes out all calendar notes from the in-memory list
    CalendarNotesList to the application's initialization file (SaveToFile.ini).
    All calendar notes are stored in the .ini file under the key 'DateNotes',
    with the integer portion of the TDateTime as the key, and the actual note
    as the value.  A sample SaveToFile.ini file:

    [DateNotes]
    38082=Today is April 5th, 2004
    38172=It's the 4th of July!
    }

    var
        i: integer;
        TheKey: string;
        DateNote: string;

    begin
    with AppIniFile do begin
        // Erase what's already there
        EraseSection(DATE_NOTE_SECTION);
        for i := 0 to Pred(CalendarNotesList.Count) do begin
            DateNote := Trim(CalendarNotesList[i]);
            if (DateNote <> '') and (Integer(CalendarNotesList.Objects[i]) <> 0) then begin
                TheKey := IntToStr(Integer(CalendarNotesList.Objects[i]));
                WriteString(DATE_NOTE_SECTION, TheKey, DateNote);
                end;
            end;
        end;
    end;

procedure TDemoForm.WallCalendar1DoubleClickCalendar(Sender: TObject; ACol, ARow: Integer; CellDate: TDateTime);

    var
        DateNote: string;

    begin
    (*
    If an "Extra Day" is double-clicked, I chose to set the calendar to that date.
    Conversely, you could also allow the user to edit the cell text for "Extra Days".
    *)
    if WallCalendar1.IsExtraDay(CellDate) then WallCalendar1.SetCalendarDate(CellDate)
    else begin
        WallCalendar1.GetCellTextByDate(CellDate, DateNote);
        DateNote := Trim(InputBox('Cell text for ' + DateToStr(CellDate), 'Enter new cell text:', DateNote));
        AddNoteToList(DateNote, CellDate);
        ShowNotesForCurrentMonthAndYear;
        end;
    end;

procedure TDemoForm.AddNoteToList(TheNote: string; TheCellDate: TDateTime);
    {
    This procedure attempts to add TheNote into CalendarNotesList based on
    TheCellDate.  If there is an entry already in the list for TheCellDate,
    the text is updated with TheNote.  Otherwise, an entry is added.
    }

    var
        Found: boolean;
        i: integer;
        EntryDate: integer;

    begin
    Found := false;
    // Look for an entry in our current list for TheCellDate
    for i := 0 to Pred(CalendarNotesList.Count) do begin
        EntryDate := Integer(CalendarNotesList.Objects[i]);
        if trunc(TheCellDate) = EntryDate then begin
            // Found one!  Let's update it and get out of here...
            Found := true;
            CalendarNotesList[i] := TheNote;
            break;
            end;
        end;

    // Couldn't find it in the list, so add it
    if not Found then CalendarNotesList.AddObject(TheNote, TObject(trunc(TheCellDate)));
    end;

procedure TDemoForm.ClearButtonClick(Sender: TObject);
    {
    This procedure clears the currently selected cell, deletes the entry from
    our internal list, and finally refreshes the calendar.
    }

    var
        i: integer;
        EntryDate: integer;
        SelectedDate: TDateTime;

    begin
    WallCalendar1.ClearCalendar;
    try
        // Look for an entry in our current list for TheCellDate
        for i := 0 to Pred(CalendarNotesList.Count) do begin
            EntryDate := Integer(CalendarNotesList.Objects[i]);
            WallCalendar1.GetCurrentCalendarDate(SelectedDate);
            if trunc(SelectedDate) = EntryDate then begin
                // Found an entry, so delete it from our internal list
                CalendarNotesList.Delete(i);
                break;
                end;
            end;
    finally
        ShowNotesForCurrentMonthAndYear;
        WallCalendar1.Refresh;
        end;
    end;

procedure TDemoForm.ShowNotesForCurrentMonthAndYear;
    (*
    Since the SetCellTextByDate() method only works for the current calendar
    month, go through our list, see if any entries are for this year and month,
    and show them.
    *)

    var
        i: integer;
        EntryDate: integer;
        TheYear, TheMonth, TheDay: word;

    begin
    // Look for an entry in our list for the current calendar month and year
    for i := 0 to Pred(CalendarNotesList.Count) do begin
        EntryDate := Integer(CalendarNotesList.Objects[i]);
        DecodeDate(EntryDate, TheYear, TheMonth, TheDay);
        if (TheYear = WallCalendar1.CalendarYear) and (TheMonth = Succ(Ord(WallCalendar1.CalendarMonth))) then begin
            WallCalendar1.SetCellTextByDate(EntryDate, CalendarNotesList[i]);
            end;
        end;
    end;

procedure TDemoForm.WallCalendar1ChangeMonth(Sender: TObject; PreviousMonth: TCalendarMonth);

    begin
    ShowNotesForCurrentMonthAndYear;
    end;

procedure TDemoForm.WallCalendar1ChangeYear(Sender: TObject; PreviousYear: Integer);

    begin
    ShowNotesForCurrentMonthAndYear;
    end;

procedure TDemoForm.WallCalendar1DecreaseCalendar(Sender: TObject; PreviousMonth: TCalendarMonth; PreviousYear: Integer);

    begin
    ShowNotesForCurrentMonthAndYear;
    end;

procedure TDemoForm.WallCalendar1IncreaseCalendar(Sender: TObject; PreviousMonth: TCalendarMonth; PreviousYear: Integer);

    begin
    ShowNotesForCurrentMonthAndYear;
    end;

procedure TDemoForm.SaveButtonClick(Sender: TObject);

    begin
    SaveCalendarNotesToIniFile;
    ShowMessage(Format('Calendar notes successfully saved to %s.', [ChangeFileExt(ParamStr(0), '.ini')]));
    end;

procedure TDemoForm.CloseButtonClick(Sender: TObject);

    begin
    Close;
    end;

end.
